<?php

namespace Fir\Controllers;

use Fir\Models;
use Fir\Views;
use Fir\Languages\Language as Language;

/**
 * The base Controller upon which all the other controllers are extended on
 */
class Controller
{
    /**
     * The database connection
     * @var    object
     */
    public $db;

    /**
     * The site settings from the DB
     * @var    array
     */
    protected $settings;

    /**
     * The view object to be passed to the controllers
     * @var    object
     */
    protected $view;

    /**
     * The language array to be passed to the controllers and views
     * @var array
     */
    protected $lang;

    /**
     * The list of available languages
     * @var array
     */
    protected $languages;

    /**
     * User selected language
     * @var string
     */
    protected $language;

    /**
     * The current URL path (route) array to be passed to the controllers
     * @var array
     */
    protected $url;

    /**
     * Currency Symbol
     * @var array
     */
    protected $currency;

    /**
     * Theme Array
     * @var array
     */
    protected $theme;

    /**
     * Controller constructor.
     * @param $db
     * @param $url
     */
    public function __construct($db, $url)
    {
        $this->db = $db;
        $this->url = $url;

        // Set the site settings
        $settings = $this->model('Settings');
        $this->settings = $settings->get();
		
		/* Use Theme Model */
		$themeModel = $this->model('Theme');
        $this->theme = $themeModel->get($this->settings['theme']);

        // Set the timezone
        if (!empty($this->settings['timezone'])) {
            date_default_timezone_set($this->settings['timezone']);
        }

        // Instantiate the Language system and set the default language
        $language = new Language();

        $this->lang = $language->set($this->settings['language']);
        $this->languages = $language->languages;
        $this->language = $language->get();

        // Instantiate the View
        $this->view = new Views\View($this->settings, $this->lang, $this->url, $this->currency, $this->theme);
    }

    /**
     * Get and instantiate the requested model
     * @param   $model  string  The model to instantiate
     * @return  object
     */
    public function model($model)
    {
        require_once(__DIR__ . '/../models/' . $model . '.php');

        // The namespace\class must be defined in a string as it can't be called shorted using new namespace\$var
        $class = 'Fir\Models\\' . $model;

        return new $class($this->db);
    }

    /**
     * Get and instantiate the requested library
     * @param   $library  string  The library to instantiate
     * @return  object
     */
    public function library($library)
    {
        require_once(__DIR__ . '/../libraries/' . $library . '.php');

        // The namespace\class must be defined in a string as it can't be called shorted using new namespace\$var
        $class = 'Fir\Libraries\\' . $library;
        
        if($library === "Validator"){
            return new $class($this->db,$this->lang);
        }

        return new $class($this->db);
    }

    /**
     * Output the final view to the user based on the request type
     *
     * @param   $data   array   The output generated by the controllers
     */
    public function run($data = null)
    {
        
        if($this->url[0] == 'cps') {

			    $data['url'] = $this->url[1];
				$data['navigation_view'] = $this->getAdminNavigation();
				$data['sidenav_view'] = $this->getAdminSidenav();
				$data['content_view'] = $data['content'];
				$data['footer_view'] = $this->getAdminFooter();
				$data['scripts_view'] = $this->getAdminScripts();
				$data['lang'] = $this->language;
            
			if (isAjax()) {
				echo json_encode(['title' => $this->view->docTitle(), 'navigation' => $data['navigation_view'], 'content' => $data['content_view'], 'footer' => $data['footer_view'], 'scripts_view' => $data['scripts_view']]);
			}else {
				echo $this->view->render($data, 'wrapper-admin');
			}
        }
        elseif($this->url[0] == 'requests') {
            if(isAjax()) {
                echo json_encode($data);
            } else {
                redirect();
            }
        } else {
            
			if($this->url[0] == 'user') :
			    $data['url'] = $this->url[0];
				$data['header_view'] = $this->getHeader();
				$data['sidenav_view'] = $this->getSidenav();
				$data['content_view'] = $data['content'];
				$data['footer_view'] = $this->getFooter();
				$data['scripts_view'] = $this->getUserScripts();
				$data['lang'] = $this->language;
			elseif($this->url[0] == CANDIDATE_URL) :
			    $data['url'] = $this->url[1];
				$data['header_view'] = $this->getHeader();
				$data['sidenav_view'] = $this->getCandidateSidenav();
				$data['content_view'] = $data['content'];
				$data['footer_view'] = $this->getFooter();
				$data['scripts_view'] = $this->getCandidateScripts();
				$data['lang'] = $this->language;
			else:	
			    $data['url'] = $this->url[0];
				$data['header_view'] = $this->getHeader();
				$data['content_view'] = $data['content'];
				$data['footer_view'] = $this->getFooter();
				$data['scripts_view'] = $this->getHomeScripts();
				$data['lang'] = $this->language;
			endif;
			if (isAjax()) {
				echo json_encode(['title' => $this->view->docTitle(), 'header' => $data['header_view'], 'content' => $data['content_view'], 'footer' => $data['footer_view'], 'scripts_view' => $data['scripts_view']]);
			}elseif($this->url[0] == 'user'){	
				echo $this->view->render($data, 'wrapper-user');			
            }elseif($this->url[0] == CANDIDATE_URL){	
				echo $this->view->render($data, 'wrapper-candidate');
			} else {
				echo $this->view->render($data, 'wrapper');
			}
        }
    }

    /**
     * This is the method from where you can pass data to the Header view
     *
     * @return string
     */
    private function getHeader()
    {
        $data = [
            'languages_list'    => $this->languages,
            'language'          => $this->language
        ];
		
		$data['url'] = $this->url[0];;
		
		/*Use User Library*/
        $user = $this->library('User');
		$data['user'] = $user->data();
		$data['user_isloggedin'] = $user->isLoggedIn(); 

        return $this->view->render($data, 'shared/header');
    }
    
    private function getFooter()
    {
        /*Use User Library*/
        $user = $this->library('User');
		$data['user'] = $user->data();
		$data['url'] = $this->url[0];
        return $this->view->render($data, 'shared/footer');
    }
    
    private function getSidenav()
    {
        $data = [];
		$data['url'] = $this->url[1];
		
		/*Use User Library*/
        $user = $this->library('User');
		$data['user'] = $user->data();
		$data['user_isloggedin'] = $user->isLoggedIn(); 
		
        return $this->view->render($data, 'user/template/sidenav');
    }
    
    private function getHomeScripts()
    {
        $data = [];
        return $this->view->render($data, 'home/template/scripts');
    }
    
    private function getUserScripts()
    {
        $data = [];
        return $this->view->render($data, 'user/template/scripts');
    }

    /**
     *
     * Admin Section
     * 
     */
    private function getLoginHeader()
    {
        $data = [
            'languages_list'    => $this->languages,
            'language'          => $this->language
        ];
		
		/* Use Settings Model */
		$settingsModel = $this->model('Settings');
        $data['settings'] = $settingsModel->get();

        return $this->view->render($data, 'shared/login_header');
    }
    
    private function getLoginFooter()
    {
        $data = [];
        return $this->view->render($data, 'shared/login_footer');
    }

    private function getAdminNavigation()
    {
        $data = [
            'languages_list'    => $this->languages,
            'language'          => $this->language
        ];
		
		$data['url'] = $this->url[1];
		
		/*Use Admin Library*/
        $admin = $this->library('Admin');
		$data['admin'] = $admin->data();
		

        return $this->view->render($data, 'admin/template/navigation');
    }
    
    private function getAdminSidenav()
    {
        $data = [];
		$data['url'] = $this->url[1];
        $admin = $this->library('Admin');
		$data['admin'] = $admin->data();
		
		/* Use Settings Model */
		$settingsModel = $this->model('Settings');
        $data['settings'] = $settingsModel->get();
		
		/* Use Admin Model */
        $adminModel = $this->model('Admin');;
		$data['has_start'] = $adminModel->has_start();
		$data['language'] = $this->language;
		
		
        return $this->view->render($data, 'admin/template/sidenav');
    }
    
    private function getAdminFooter()
    {
        $data = [];
        return $this->view->render($data, 'admin/template/footer');
    }
    
    private function getAdminScripts()
    {
        $data = [];
        return $this->view->render($data, 'admin/template/scripts');
    }
    
    
    private function getCandidateSidenav()
    {
        $data = [];
		$data['url'] = $this->url[1];
		
		/*Use User Library*/
        $user = $this->library('User');
		$data['user'] = $user->data();
		$data['user_isloggedin'] = $user->isLoggedIn(); 

		
        return $this->view->render($data, 'candidate/template/sidenav');
    }
    
    private function getCandidateScripts()
    {
        $data = [];
        return $this->view->render($data, 'candidate/template/scripts');
    }
}