<?php

namespace MessageBird\Objects\Conversation;

use JsonSerializable;
use MessageBird\Objects\Base;

/**
 * Webhooks enable real-time notifications of conversation events to be
 * delivered to endpoints on your own server. A webhook can be configured with
 * a URL and a list of events that should be subscribed to for notifications.
 * It's possible to create multiple webhooks with different URLs to listen to
 * one or more events each.
 */
class Webhook extends Base implements JsonSerializable
{
    const EVENT_CONVERSATION_CREATED = 'conversation.created';
    const EVENT_CONVERSATION_UPDATED = 'conversation.updated';
    const EVENT_MESSAGE_CREATED = 'message.created';
    const EVENT_MESSAGE_UPDATED = 'message.updated';

    /**
     * A unique ID generated by the MessageBird platform that identifies this
     * webhook.
     * 
     * @var string
     */
    public $id;

    /**
     * The URL of this webhook object.
     * 
     * @var string
     */
    public $href;

    /**
     * The unique identifier for a MessageBird channel taht this webhook will
     * subscribe to events for.
     * 
     * @var string
     */
    public $channelId;

    /**
     * A list of event name strings from the list of available events that
     * trigger this webhook. Possible values: "conversation.created",
     * "conversation.updated", "message.created", "message.updated".
     *
     * @var array
     */
    public $events;

    /**
     * The endpoint URL that requests are sent to.
     * 
     * @var string
     */
    public $url;

    /**
     * The date and time when this webhook was first created in RFC3339 format.
     *
     * @var string
     */
    public $createdDatetime;

    /**
     * The date and time when this webhook was last updated in RFC3339 format.
     *
     * @var string
     */
    public $updatedDatetime;

    /**
     * Serialize only non empty fields.
     */
    public function jsonSerialize()
    {
        $json = array();
        
        foreach (get_object_vars($this) as $key => $value) {
            if (!empty($value)) {
                $json[$key] = $value;
            }
        }

        return $json;
    }
}
